#include "general.h"
#include <maxheapdirect.h>
#include <notify.h>
#include "w3dutilities.h"

namespace
{
	struct NodeCloneInfo //See notify.h in the max SDK. NOTIFY_POST_NODES_CLONED
	{
		INodeTab* OriginalNodes = nullptr;
		INodeTab* NewNodes = nullptr;
	};
}

namespace W3D::MaxTools
{
	W3DUtilities::W3DUtilities()
		: m_SelectedNodes()
		, m_ExportSettings(*this)
		, m_MiscUtilities(*this)
	{ }

	void W3DUtilities::BeginEditParams(Interface * ip, IUtil * iu)
	{
		m_ExportSettings.Initialise(ip);
		m_MiscUtilities.Initialise(ip);
		SelectionSetChanged(ip, iu);
	}

	void W3DUtilities::EndEditParams(Interface * ip, IUtil * iu)
	{
		m_ExportSettings.Close(ip);
		m_MiscUtilities.Close(ip);
	}

	void W3DUtilities::SelectionSetChanged(Interface *ip, IUtil *iu)
	{
		m_SelectedNodes.clear();
		m_SelectedNodes.reserve(ip->GetSelNodeCount());

		for (int i = 0; i < ip->GetSelNodeCount(); ++i)
			m_SelectedNodes.push_back(ip->GetSelNode(i));

		m_ExportSettings.RefreshUI();
	}

	void W3DUtilities::DeleteThis()
	{
		delete this;
	}

	W3DAppDataChunk&  W3DUtilities::GetOrCreateW3DAppDataChunk(INode& node)
	{
		return *static_cast<W3DAppDataChunk*>(GetOrCreateExportSettingsChunk(node).data);
	}

	AppDataChunk & W3DUtilities::GetOrCreateChunk(INode& node, W3D::MaxTools::W3DUtilityAppDataSubID subID)
	{
		const int chunkSID = enum_to_value(subID);

		AppDataChunk* chunk = node.GetAppDataChunk(W3DUtilitiesClassDesc::Instance()->ClassID(), W3DUtilitiesClassDesc::Instance()->SuperClassID(), chunkSID);

		if (!chunk)
		{
			node.AddAppDataChunk(W3DUtilitiesClassDesc::Instance()->ClassID(), W3DUtilitiesClassDesc::Instance()->SuperClassID(), chunkSID, 0, nullptr);
			chunk = node.GetAppDataChunk(W3DUtilitiesClassDesc::Instance()->ClassID(), W3DUtilitiesClassDesc::Instance()->SuperClassID(), chunkSID);
		}

		return *chunk;
	}

	AppDataChunk & W3DUtilities::GetOrCreateExportSettingsChunk(INode& node)
	{
		AppDataChunk& chunk = GetOrCreateChunk(node, W3DUtilityAppDataSubID::ExportSettings);
		if (!chunk.data)
		{
			chunk.length = sizeof(W3DAppDataChunk);
			void* alloc = MAX_malloc(chunk.length);
			chunk.data = new(alloc) W3DAppDataChunk;
		}

		return chunk;
	}

	ClassDesc2 * W3DUtilitiesClassDesc::Instance()
	{
		static W3DUtilitiesClassDesc s_instance;
		return &s_instance;
	}

	W3DUtilitiesClassDesc::W3DUtilitiesClassDesc()
	{
		RegisterNotification(NotifyPostNodesCloned, this, NOTIFY_POST_NODES_CLONED);
	}

	W3DUtilitiesClassDesc::~W3DUtilitiesClassDesc()
	{
		UnRegisterNotification(NotifyPostNodesCloned, this, NOTIFY_POST_NODES_CLONED);
	}

	void W3DUtilitiesClassDesc::NotifyPostNodesCloned(void * param, NotifyInfo * info)
	{
		NodeCloneInfo* cloneInfo = static_cast<NodeCloneInfo*>(info->callParam);

		for (int i = 0; i < cloneInfo->NewNodes->Count(); ++i)
		{
			INode& orig = *(*cloneInfo->OriginalNodes)[i];
			INode& clone = *(*cloneInfo->NewNodes)[i];

			for (int subIdx = 0; subIdx < enum_to_value(W3DUtilityAppDataSubID::Num); ++subIdx)
			{
				AppDataChunk* origChunk = orig.GetAppDataChunk(Instance()->ClassID(), Instance()->SuperClassID(), subIdx);
				if (origChunk && origChunk->data)
				{
					clone.AddAppDataChunk(Instance()->ClassID(), Instance()->SuperClassID(), subIdx, origChunk->length, MAX_malloc(origChunk->length));
					AppDataChunk* newChunk = clone.GetAppDataChunk(Instance()->ClassID(), Instance()->SuperClassID(), subIdx);
					memcpy(newChunk->data, origChunk->data, newChunk->length);
				}
			}
		}
	}
}